﻿namespace Microsoft.Samples.PlanMyNight.Web.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.ComponentModel.DataAnnotations;
    using System.Diagnostics;
    using System.Globalization;
    using System.Linq;
    using System.Text;
    using System.Web.Mvc;
    using System.Web.UI;
    using Microsoft.Samples.PlanMyNight.Infrastructure.Mvc;
    using Microsoft.Samples.PlanMyNight.Data;
    using Microsoft.Samples.PlanMyNight.Entities;
    using Microsoft.Samples.PlanMyNight.Infrastructure;
    using Microsoft.Samples.PlanMyNight.Web.Properties;
    using Microsoft.Samples.PlanMyNight.Web.ViewModels;
    using System.Web.Routing;

    [HandleErrorWithContentType()]
    public class SearchController : Controller
    {
        public const int DefaultPageSize = 8;

        public const int MaxAutoCompleteItems = 30;

        public const double MetersInAMile = 1609.344;

        private readonly IItinerariesRepository itinerariesRepository;

        private readonly IReferenceRepository referenceRepository;

        private readonly IActivitiesRepository activitiesRepository;

        private readonly IMembershipService membershipService;

        public SearchController() :
            this(
                new ServiceFactory().GetMembershipService(),
                new ServiceFactory().GetReferenceRepositoryInstance(),
                new ServiceFactory().GetActivitiesRepositoryInstance(),
                new ServiceFactory().GetItinerariesRepositoryInstance())
        {
        }

        public SearchController(IMembershipService membershipService, IReferenceRepository referenceRepository, IActivitiesRepository activitiesRepository, IItinerariesRepository itinerariesRepository)
        {
            this.membershipService = membershipService;
            this.itinerariesRepository = itinerariesRepository;
            this.referenceRepository = referenceRepository;
            this.activitiesRepository = activitiesRepository;
        }

        public ActionResult Index()
        {
            BaseSearchQuery searchCriteria = null;

            // if not in natural mode
            if (Request.QueryString["m"] != "n")
            {
                // default - using user profile for first time search
                var profile = this.membershipService.GetCurrentProfile();
                if (profile != null)
                {
                    // set defaults from user profile
                    if (!string.IsNullOrEmpty(profile.State))
                    {
                        // preselect state & city from profile
                        searchCriteria = new AdvancedSearchQuery
                        {
                            Type = SearchType.Activity,
                            State = profile.State,
                            City = profile.City,
                            ActivityTypeId = profile.PreferredActivityTypeId.Value
                        };

                        // state, city and preferredActivityType found, generate search
                        if (!string.IsNullOrEmpty(profile.City) && (profile.PreferredActivityTypeId ?? 0) != 0)
                        {
                            return this.Advanced((AdvancedSearchQuery)searchCriteria);
                        }
                    }
                }
            }

            // empty search form
            var model = new SearchViewModel
            {
                AdvancedMode = Request.QueryString["m"] == "a",
                Criteria = searchCriteria ?? new NaturalSearchQuery(),
                SearchFields = this.RetrieveSearchFields(searchCriteria as AdvancedSearchQuery)
            };

            return View("Index", model);
        }

        [JsonCache(Duration = 900, Location = OutputCacheLocation.Any, VaryByParam = "*", VaryByHeader = "Content-Type")]
        public ActionResult Search(NaturalSearchQuery criteria)
        {
            // paging
            criteria.PageSize = DefaultPageSize;
            if (criteria.Page <= 0) criteria.Page = 1;

            // validate query
            bool validationPassed = ModelState.IsValid;
            if (validationPassed && !criteria.IsActivityValid(this.activitiesRepository))
            {
                ModelState.AddModelError("Query", "The provided activity is not valid.");
                validationPassed = false;
            }

            if (criteria.Type == SearchType.ActivityItinerary)
                criteria.Type = SearchType.Activity;

            // viewmodel
            var model = new SearchViewModel
            {
                AdvancedMode = false,
                Criteria = criteria,
                CriteriaDescription = validationPassed ? this.GetCriteriaDescription(criteria, " | ") : string.Empty,
                KeywordsMetatag = validationPassed ? this.GetCriteriaDescription(criteria, ", ") : string.Empty
            };

            if (validationPassed)
            {
                Trace.WriteLine("Starting search...");
                Stopwatch timer = new Stopwatch();
                timer.Start();

                switch (criteria.Type)
                {
                    case SearchType.Activity:
                        var pagedActivities = this.activitiesRepository.Search(criteria);
                        FillViewModel(model, pagedActivities);

                        break;

                    case SearchType.Itinerary:
                        var location = this.ParseLocation(criteria);
                        if (location != null)
                        {
                            var pagedItineraries = this.itinerariesRepository.SearchByCity(criteria.ActivityTypeId, location.State, location.City, DefaultPageSize, criteria.Page);
                            FillViewModel(model, pagedItineraries);
                        }

                        break;
                }

                timer.Stop();
                Trace.WriteLine("Search completed! " + timer.ElapsedMilliseconds + "ms");
            }

            if (this.IsAjaxCall())
            {
                // json
                return new JsonResult { JsonRequestBehavior = JsonRequestBehavior.AllowGet, Data = model };
            }

            model.SearchFields = this.RetrieveSearchFields();
            return View("Index", model);
        }

        [JsonCache(Duration = 900, Location = OutputCacheLocation.Any, VaryByParam = "*", VaryByHeader = "Content-Type")]
        public ActionResult Advanced(AdvancedSearchQuery criteria)
        {
            // paging
            criteria.PageSize = DefaultPageSize;
            if (criteria.Page <= 0) criteria.Page = 1;
            if (criteria.Type == SearchType.ActivityItinerary)
                criteria.Type = SearchType.Activity;

            // validate query
            var validationPassed = ModelState.IsValid;

            // viewmodel
            var model = new SearchViewModel
            {
                AdvancedMode = true,
                Criteria = criteria,
                CriteriaDescription = validationPassed ? this.GetCriteriaDescription(criteria, " | ") : string.Empty,
                KeywordsMetatag = validationPassed ? this.GetCriteriaDescription(criteria, ", ") : string.Empty
            };

            if (validationPassed)
            {
                Trace.WriteLine("Starting search...");
                Stopwatch timer = new Stopwatch();
                timer.Start();

                switch (criteria.Type)
                {
                    case SearchType.Activity:
                        var pagedActivities = this.activitiesRepository.Search(criteria);
                        FillViewModel(model, pagedActivities);
                        break;

                    case SearchType.Itinerary:

                        PagingResult<Itinerary> pagedItineraries = null;

                        if (!string.IsNullOrWhiteSpace(criteria.State) && !string.IsNullOrWhiteSpace(criteria.City) && !string.IsNullOrWhiteSpace(criteria.StreetAddress) && criteria.Radius.HasValue)
                        {
                            // by state + city + address + radius (spatial search)
                            var address = new ActivityAddress
                            {
                                StreetAddress = criteria.StreetAddress,
                                City = criteria.City,
                                State = criteria.State,
                                Zip = criteria.Zip
                            };

                            var radiusInMeters = criteria.Radius.Value * MetersInAMile;
                            var location = this.activitiesRepository.GeocodeAddress(address);
                            pagedItineraries = this.itinerariesRepository.SearchByRadius(criteria.ActivityTypeId, location.Item1, location.Item2, radiusInMeters, DefaultPageSize, criteria.Page);
                        }
                        else if (!string.IsNullOrWhiteSpace(criteria.Zip))
                        {
                            // by zip code
                            pagedItineraries = this.itinerariesRepository.SearchByZipCode(criteria.ActivityTypeId, criteria.Zip, DefaultPageSize, criteria.Page);
                        }
                        else if (!string.IsNullOrWhiteSpace(criteria.State) && !string.IsNullOrWhiteSpace(criteria.City))
                        {
                            // by state + city
                            pagedItineraries = this.itinerariesRepository.SearchByCity(criteria.ActivityTypeId, criteria.State, criteria.City, DefaultPageSize, criteria.Page);
                        }

                        FillViewModel(model, pagedItineraries);

                        break;
                }

                timer.Stop();
                Trace.WriteLine("Search completed! " + timer.ElapsedMilliseconds + "ms");
            }

            if (this.IsAjaxCall())
            {
                // json
                return new JsonResult { JsonRequestBehavior = JsonRequestBehavior.AllowGet, Data = model };
            }

            model.SearchFields = this.RetrieveSearchFields(criteria);

            return View("Index", model);
        }

        [JsonCache(Duration = 900, Location = OutputCacheLocation.Any, VaryByParam = "*", VaryByHeader = "Content-Type")]
        public ActionResult ByActivity(string activityId, int? page)
        {
            var activity = this.activitiesRepository.RetrieveActivity(activityId);
            if (activity == null)
            {
                return Redirect("~/");
            }

            var pageNumber = page ?? 1;
            var searchCriteria = new AdvancedSearchQuery
            {
                Type = SearchType.ActivityItinerary,
                State = activity.State,
                City = activity.City,
                ActivityTypeId = activity.TypeId
            };

            // viewmodel
            var model = new SearchViewModel
            {
                SearchActivity = activity,
                CriteriaDescription = "Itineraries for '" + activity.Name + "'",
                Criteria = searchCriteria
            };

            var results = this.itinerariesRepository.SearchByActivity(activityId, DefaultPageSize, pageNumber);
            FillViewModel(model, results);

            if (this.IsAjaxCall())
            {
                // json
                return new JsonResult() { JsonRequestBehavior = JsonRequestBehavior.AllowGet, Data = model };
            }

            model.SearchFields = this.RetrieveSearchFields();
            return View("Index", model);
        }

        [JsonCache(Duration = int.MaxValue, Location = OutputCacheLocation.Any, VaryByParam = "state;q", VaryByHeader = "Content-Type")]
        public JsonResult ZipCodes(string state, string q)
        {
            var zipCodes = this.referenceRepository.RetrieveZipCodes(state);
            var match = zipCodes.Where(z => z.StartsWith(q, StringComparison.OrdinalIgnoreCase)).Take(MaxAutoCompleteItems);
            return new JsonResult
            {
                JsonRequestBehavior = JsonRequestBehavior.AllowGet,
                Data = match
            };
        }

        [JsonCache(Duration = int.MaxValue, Location = OutputCacheLocation.Any, VaryByParam = "state;q")]
        public JsonResult Cities(string state, string q)
        {
            var cities = this.referenceRepository.RetrieveCities(state);
            var match = cities.Where(z => z.StartsWith(q, StringComparison.OrdinalIgnoreCase)).Take(MaxAutoCompleteItems);
            return new JsonResult
            {
                JsonRequestBehavior = JsonRequestBehavior.AllowGet,
                Data = match
            };
        }

        private void FillViewModel<T>(SearchViewModel model, PagingResult<T> results)
        {
            if (results != null)
            {
                // items
                if (typeof(T) == typeof(Activity))
                {
                    model.Items = results.Items.Select(e => this.ToViewModel(e as Activity)).ToArray();
                }
                else if (typeof(T) == typeof(Itinerary))
                {
                    model.Items = results.Items.Select(e => this.ToViewModel(e as Itinerary)).ToArray();
                }

                // paging
                model.CurrentPage = results.CurrentPage;
                model.TotalPages = results.TotalPages;

                // paging links
                var urlHelper = new UrlHelper(this.ControllerContext.RequestContext, RouteTable.Routes);
                if (model.SearchActivity == null)
                {
                    // Natural or Advanced
                    var actionName = model.Criteria.GetType() == typeof(NaturalSearchQuery) ? "Search" : "Advanced";
                    if (results.CurrentPage < results.TotalPages)
                        model.NextPageLink = urlHelper.Action(actionName, model.Criteria.NewPage(results.CurrentPage + 1).GetRouteValues());
                    if (results.CurrentPage > 1)
                        model.PreviousPageLink = urlHelper.Action(actionName, model.Criteria.NewPage(results.CurrentPage - 1).GetRouteValues());
                }
                else
                {
                    // ByActivity
                    if (results.CurrentPage < results.TotalPages)
                        model.NextPageLink = this.BuildUrlFromExpression<SearchController>(c => c.ByActivity(model.SearchActivity.Id, model.CurrentPage + 1));
                    if (results.CurrentPage > 1)
                        model.NextPageLink = this.BuildUrlFromExpression<SearchController>(c => c.ByActivity(model.SearchActivity.Id, model.CurrentPage - 1));
                }
            }
        }

        private ActivityAddress ParseLocation(NaturalSearchQuery criteria)
        {
            var location = this.activitiesRepository.ParseQueryLocation(criteria.Query);
            if (location != null)
            {
                if (location.State.Length > 2)
                {
                    // use abbreviated state
                    var state = this.referenceRepository.RetrieveStates().SingleOrDefault(a => a.Name.Equals(location.State, StringComparison.OrdinalIgnoreCase));
                    if (state != null)
                    {
                        location.State = state.Abbreviation;
                    }
                }

                return location;
            }

            return null;
        }

        private SearchFields RetrieveSearchFields()
        {
            return this.RetrieveSearchFields(null);
        }

        private SearchFields RetrieveSearchFields(AdvancedSearchQuery currentCriteria)
        {
            var types = this.activitiesRepository.RetrieveActivityTypes()
                                                 .Select(o => new SelectListItem { Text = o.Name, Value = o.Id.ToString(), Selected = (currentCriteria != null && o.Id == currentCriteria.ActivityTypeId) })
                                                 .ToList();
            types.Insert(0, new SelectListItem { Text = "Select...", Value = "0" });

            var states = this.referenceRepository.RetrieveStates()
                                                 .Select(o => new SelectListItem { Text = o.Name, Value = o.Abbreviation, Selected = (currentCriteria != null && o.Abbreviation == currentCriteria.State) })
                                                 .ToList();
            states.Insert(0, new SelectListItem { Text = "Any state", Value = string.Empty });

            var miles = new List<SelectListItem>();
            miles.Add(new SelectListItem { Text = "Any radius", Value = "" });
            miles.Add(new SelectListItem { Text = "1", Value = "1" });
            miles.Add(new SelectListItem { Text = "2", Value = "2" });
            miles.Add(new SelectListItem { Text = "5", Value = "5" });
            miles.Add(new SelectListItem { Text = "10", Value = "10" });
            miles.Add(new SelectListItem { Text = "15", Value = "15" });

            return new SearchFields
            {
                ActivityTypes = types,
                States = states,
                Miles = miles,
                City = currentCriteria != null ? currentCriteria.City : string.Empty
            };
        }

        private string GetCriteriaDescription(AdvancedSearchQuery searchCriteria, string separator = " | ")
        {
            StringBuilder title = new StringBuilder();
            if (searchCriteria.ActivityTypeId > 0)
            {
                title.Append(this.activitiesRepository.RetrieveActivityTypes().Where(a => a.Id == searchCriteria.ActivityTypeId).Select(a => a.PluralizedName).FirstOrDefault());
                title.Append(separator);
            }

            if (!string.IsNullOrEmpty(searchCriteria.City))
            {
                title.Append(searchCriteria.City);
                title.Append(separator);
            }

            if (!string.IsNullOrEmpty(searchCriteria.State))
            {
                title.Append(searchCriteria.State);
                title.Append(separator);
            }

            if (!string.IsNullOrEmpty(searchCriteria.Zip))
            {
                title.Append(searchCriteria.Zip);
                title.Append(separator);
            }

            title.AppendFormat(CultureInfo.CurrentUICulture, "Page {0}", searchCriteria.Page);

            return title.ToString();
        }

        private string GetCriteriaDescription(NaturalSearchQuery searchCriteria, string separator = " | ")
        {
            StringBuilder title = new StringBuilder();

            if (!string.IsNullOrEmpty(searchCriteria.Query))
            {
                title.Append(searchCriteria.Query);
                title.Append(separator);
            }

            title.AppendFormat(CultureInfo.CurrentUICulture, "Page {0}", searchCriteria.Page);

            return title.ToString();
        }
    }
}
